/*
 * Decompiled with CFR 0.152.
 */
package org.apache.james.user.lib.model;

import com.google.common.base.Preconditions;
import com.google.common.base.Splitter;
import com.google.common.collect.ImmutableList;
import java.io.ByteArrayOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.nio.charset.StandardCharsets;
import java.security.MessageDigest;
import java.security.NoSuchAlgorithmException;
import java.security.spec.InvalidKeySpecException;
import java.util.Arrays;
import java.util.List;
import java.util.Objects;
import java.util.Optional;
import javax.crypto.SecretKeyFactory;
import javax.crypto.spec.PBEKeySpec;
import javax.mail.MessagingException;
import javax.mail.internet.MimeUtility;

public class Algorithm {
    private final String rawValue;
    private final HashingMode hashingMode;
    private final Hasher hasher;

    public static Algorithm of(String algorithmName) {
        return Algorithm.of(algorithmName, HashingMode.PLAIN);
    }

    public static Algorithm of(String algorithmName, String fallbackHashingMode) {
        return Algorithm.of(algorithmName, HashingMode.parse(fallbackHashingMode));
    }

    public static Algorithm of(String algorithmName, HashingMode fallbackHashingMode) {
        List spec = Splitter.on((char)'/').splitToList((CharSequence)algorithmName);
        if (spec.size() == 1) {
            return new Algorithm(algorithmName, fallbackHashingMode);
        }
        return new Algorithm((String)spec.get(0), HashingMode.parse((String)spec.get(1)));
    }

    private Algorithm(String rawValue, HashingMode hashingMode) {
        this.rawValue = rawValue;
        this.hashingMode = hashingMode;
        this.hasher = Hasher.from(this);
    }

    public String asString() {
        return this.rawValue + "/" + this.hashingMode.name().toLowerCase();
    }

    public String getName() {
        return this.rawValue;
    }

    public String getHashingMode() {
        return this.hashingMode.name();
    }

    public boolean isLegacy() {
        return this.hashingMode == HashingMode.LEGACY || this.hashingMode == HashingMode.LEGACY_SALTED;
    }

    public boolean isPBKDF2() {
        return this.hasher instanceof PBKDF2Hasher;
    }

    public boolean isSalted() {
        return this.hashingMode == HashingMode.SALTED || this.hashingMode == HashingMode.LEGACY_SALTED;
    }

    public Hasher hasher() {
        return this.hasher;
    }

    public String digest(String pass, String salt) {
        try {
            return this.encodeInBase64(this.hasher().digestString(pass, salt));
        }
        catch (IOException | NoSuchAlgorithmException | InvalidKeySpecException | MessagingException e) {
            throw new RuntimeException("Fatal error when hashing password", e);
        }
    }

    public String encodeInBase64(byte[] digest) throws MessagingException, IOException {
        ByteArrayOutputStream bos = new ByteArrayOutputStream();
        OutputStream encodedStream = MimeUtility.encode((OutputStream)bos, (String)"base64");
        encodedStream.write(digest);
        if (!this.isLegacy()) {
            encodedStream.close();
        }
        return bos.toString(StandardCharsets.ISO_8859_1);
    }

    public final boolean equals(Object o) {
        if (o instanceof Algorithm) {
            Algorithm that = (Algorithm)o;
            return Objects.equals(this.rawValue, that.rawValue) && Objects.equals((Object)this.hashingMode, (Object)that.hashingMode);
        }
        return false;
    }

    public final int hashCode() {
        return Objects.hash(new Object[]{this.rawValue, this.hashingMode});
    }

    public static enum HashingMode {
        PLAIN,
        SALTED,
        LEGACY,
        LEGACY_SALTED;


        public static HashingMode parse(String value) {
            return Arrays.stream(HashingMode.values()).filter(aValue -> value.equalsIgnoreCase(aValue.toString())).findFirst().orElseThrow(() -> new IllegalArgumentException("Unsupported value for HashingMode: " + value + ". Should be one of " + ImmutableList.copyOf((Object[])HashingMode.values())));
        }
    }

    public static interface Hasher {
        public static Hasher from(Algorithm algorithm) {
            return PBKDF2Hasher.from(algorithm).orElseGet(() -> new RegularHashingSpec(algorithm));
        }

        public byte[] digestString(String var1, String var2) throws IOException, NoSuchAlgorithmException, InvalidKeySpecException;
    }

    public static class PBKDF2Hasher
    implements Hasher {
        public static final int DEFAULT_ITERATION_COUNT = 1000;
        public static final int DEFAULT_KEY_SIZE = 512;
        private final int iterationCount;
        private final int keySize;

        public static Optional<Hasher> from(Algorithm algorithm) {
            if (algorithm.getName().startsWith("PBKDF2")) {
                List parts = Splitter.on((char)'-').splitToList((CharSequence)algorithm.getName());
                return Optional.of(new PBKDF2Hasher(PBKDF2Hasher.parseIterationCount(parts), PBKDF2Hasher.parseKeySize(parts)));
            }
            return Optional.empty();
        }

        private static int parseKeySize(List<String> parts) {
            if (parts.size() >= 3) {
                return Integer.parseInt(parts.get(2));
            }
            return 512;
        }

        private static int parseIterationCount(List<String> parts) {
            if (parts.size() >= 2) {
                return Integer.parseInt(parts.get(1));
            }
            return 1000;
        }

        public PBKDF2Hasher(int iterationCount, int keySize) {
            Preconditions.checkArgument((iterationCount > 0 ? 1 : 0) != 0, (Object)"'iterationCount' should be greater than 0");
            Preconditions.checkArgument((keySize > 0 ? 1 : 0) != 0, (Object)"'keySize' should be greater than 0");
            this.iterationCount = iterationCount;
            this.keySize = keySize;
        }

        @Override
        public byte[] digestString(String pass, String salt) throws NoSuchAlgorithmException, InvalidKeySpecException {
            PBEKeySpec spec = new PBEKeySpec(pass.toCharArray(), salt.getBytes(StandardCharsets.ISO_8859_1), this.iterationCount, this.keySize);
            SecretKeyFactory factory = SecretKeyFactory.getInstance("PBKDF2WithHmacSHA1");
            return factory.generateSecret(spec).getEncoded();
        }

        public final boolean equals(Object o) {
            if (o instanceof PBKDF2Hasher) {
                PBKDF2Hasher that = (PBKDF2Hasher)o;
                return Objects.equals(this.iterationCount, that.iterationCount) && Objects.equals(this.keySize, that.keySize);
            }
            return false;
        }

        public final int hashCode() {
            return Objects.hash(this.iterationCount, this.keySize);
        }
    }

    public static class RegularHashingSpec
    implements Hasher {
        private final Algorithm algorithm;

        public RegularHashingSpec(Algorithm algorithm) {
            this.algorithm = algorithm;
        }

        @Override
        public byte[] digestString(String pass, String salt) throws NoSuchAlgorithmException {
            MessageDigest md = MessageDigest.getInstance(this.algorithm.getName());
            String saltedPass = this.applySalt(this.algorithm, pass, salt);
            return md.digest(saltedPass.getBytes(StandardCharsets.ISO_8859_1));
        }

        private String applySalt(Algorithm algorithm, String pass, String salt) {
            if (algorithm.isSalted()) {
                return salt + pass;
            }
            return pass;
        }
    }
}

