/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *      http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */
package org.apache.cocoon.forms.formmodel;

import java.util.ArrayList;
import java.util.Collections;
import java.util.HashMap;
import java.util.Iterator;
import java.util.List;
import java.util.Map;

import org.apache.excalibur.xml.sax.XMLizable;

import org.apache.cocoon.forms.FormsConstants;
import org.apache.cocoon.forms.FormsException;
import org.apache.cocoon.forms.datatype.DatatypeManager;
import org.apache.cocoon.forms.event.CreateListener;
import org.apache.cocoon.forms.event.WidgetListener;
import org.apache.cocoon.forms.event.WidgetListenerBuilder;
import org.apache.cocoon.forms.expression.ExpressionManager;
import org.apache.cocoon.forms.util.DomHelper;
import org.apache.cocoon.forms.validation.WidgetValidatorBuilder;

import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.NodeList;

/**
 * Abstract base class for WidgetDefinitionBuilders. Provides functionality
 * common to many implementations.
 *
 * @version $Id: AbstractWidgetDefinitionBuilder.java 615336 2008-01-25 20:48:44Z vgritsenko $
 */
public abstract class AbstractWidgetDefinitionBuilder implements WidgetDefinitionBuilder {

    protected Map widgetDefinitionBuilders;
    protected Map widgetValidatorBuilders;
    protected Map widgetListenerBuilders;
    protected DatatypeManager datatypeManager;
    protected ExpressionManager expressionManager;


    public WidgetDefinition buildWidgetDefinition(Element widgetElement) throws Exception {
        throw new UnsupportedOperationException("Please use the other signature with WidgetDefinitionBuilderContext!");
    }

    protected void setupDefinition(Element widgetElement, AbstractWidgetDefinition definition, WidgetDefinitionBuilderContext context)
    throws Exception {
        // location
        definition.setLocation(DomHelper.getLocationObject(widgetElement));

        if (context.getSuperDefinition() != null) {
            definition.initializeFrom(context.getSuperDefinition());
        }

        setCommonProperties(widgetElement, definition);
        setValidators(widgetElement, definition);
        setCreateListeners(widgetElement, definition);
    }

    protected void setCommonProperties(Element widgetElement, AbstractWidgetDefinition widgetDefinition)
    throws Exception {

        // id
        if (widgetDefinition instanceof FormDefinition) {
            // FormDefinition is the *only* kind of widget that has an optional id
            widgetDefinition.setId(DomHelper.getAttribute(widgetElement, "id", ""));
        } else {
            String id = DomHelper.getAttribute(widgetElement, "id");
            if (id.indexOf('/') != -1 || id.indexOf('.') != -1) {
                throw new FormsException("A widget name cannot contain '.' or '/' as this conflicts with widget paths.",
                                         DomHelper.getLocationObject(widgetElement));
            }
            // NewDefinitions are allowed to have a : in their id because they can look up
            // class widgets from the library directly
            if (id.indexOf(':') != -1 && !(widgetDefinition instanceof NewDefinition)) {
                throw new FormsException("A widget name cannot contain ':' as this conflicts with library prefixes",
                                         DomHelper.getLocationObject(widgetElement));
            }
            widgetDefinition.setId(id);
        }

        // state
        String stateValue = DomHelper.getAttribute(widgetElement, "state", null);
        if (stateValue != null) {
            WidgetState state = WidgetState.stateForName(stateValue);
            if (state == null) {
                throw new FormsException("Unknown value '" + stateValue +"' for state attribute.",
                                         DomHelper.getLocationObject(widgetElement));
            }
            widgetDefinition.setState(state);
        }

        // attributes
        Element attrContainer = DomHelper.getChildElement(widgetElement, FormsConstants.DEFINITION_NS, "attributes", false);
        if (attrContainer != null) {
            // There's a <fd:attributes> element. Get its <fd:attribute> children
            Element[] attrs = DomHelper.getChildElements(attrContainer, FormsConstants.DEFINITION_NS, "attribute");
            if (attrs != null && attrs.length > 0) {
                // We actually do have some
                Map attrMap = new HashMap();
                for (int i = 0; i < attrs.length; i++) {
                    attrMap.put(DomHelper.getAttribute(attrs[i], "name"), DomHelper.getAttribute(attrs[i], "value"));
                }
                widgetDefinition.setAttributes(attrMap);
            }
        }
    }

    protected WidgetDefinition buildAnotherWidgetDefinition(Element widgetElement, WidgetDefinitionBuilderContext context)
    throws Exception {
        String widgetName = widgetElement.getLocalName();
        WidgetDefinitionBuilder builder = (WidgetDefinitionBuilder) widgetDefinitionBuilders.get(widgetName);
        if (builder == null) {
            throw new FormsException("Unknown kind of widget '" + widgetName + "'.",
                                     DomHelper.getLocationObject(widgetElement));
        }

        WidgetDefinition def = builder.buildWidgetDefinition(widgetElement, context);

        // register this class with the local library, if any.
        if (DomHelper.getAttributeAsBoolean(widgetElement, "register", false)) {
            context.getLocalLibrary().addDefinition(def);
        }

        return def;
    }

    protected List buildEventListeners(Element widgetElement, String elementName, Class listenerClass)
    throws Exception {
        List result = null;
        Element listenersElement = DomHelper.getChildElement(widgetElement, FormsConstants.DEFINITION_NS, elementName);
        if (listenersElement != null) {
            NodeList list = listenersElement.getChildNodes();
            for (int i = 0; i < list.getLength(); i++) {
                if (list.item(i).getNodeType() == Node.ELEMENT_NODE) {
                    Element listenerElement = (Element)list.item(i);
                    WidgetListenerBuilder builder = (WidgetListenerBuilder) widgetListenerBuilders.get(listenerElement.getLocalName());
                    if (builder == null) {
                        throw new FormsException("Unknown kind of eventlistener '" + listenerElement.getLocalName() + "'.",
                                                 DomHelper.getLocationObject(listenerElement));
                    }
                    WidgetListener listener = builder.buildListener(listenerElement, listenerClass);
                    if (result == null) {
                        result = new ArrayList();
                    }
                    result.add(listener);
                }
            }
        }

        return result == null ? Collections.EMPTY_LIST : result;
    }

    protected void setDisplayData(Element widgetElement, AbstractWidgetDefinition widgetDefinition)
    throws Exception {
        final String[] names = {"label", "help", "hint"};
        Map displayData = new HashMap(names.length);
        for (int i = 0; i < names.length; i++) {
            XMLizable data = null;
            Element dataElement = DomHelper.getChildElement(widgetElement, FormsConstants.DEFINITION_NS, names[i]);
            if (dataElement != null) {
                data = DomHelper.compileElementContent(dataElement);
            }

            // NOTE: We put also null values in the may in order to test their existence
            //       (see AbstractWidgetDefinition.generateDisplayData)
            displayData.put(names[i], data);
        }

        widgetDefinition.setDisplayData(displayData);
    }

    protected void setValidators(Element widgetElement, AbstractWidgetDefinition widgetDefinition)
    throws Exception {
        Element validatorElement = DomHelper.getChildElement(widgetElement, FormsConstants.DEFINITION_NS, "validation");
        if (validatorElement != null) {
            NodeList list = validatorElement.getChildNodes();
            for (int i = 0; i < list.getLength(); i++) {
                if (list.item(i).getNodeType() == Node.ELEMENT_NODE) {
                    Element element = (Element)list.item(i);
                    String name = element.getLocalName();
                    WidgetValidatorBuilder builder = (WidgetValidatorBuilder)this.widgetValidatorBuilders.get(name);
                    if (builder == null) {
                        throw new FormsException("Unknown kind of validator '" + name + "'.",
                                                 DomHelper.getLocationObject(element));
                    }

                    widgetDefinition.addValidator(builder.build(element, widgetDefinition));
                }
            }
        }
    }

    protected void setCreateListeners(Element widgetElement, AbstractWidgetDefinition widgetDefinition)
    throws Exception {
        Iterator i = buildEventListeners(widgetElement, "on-create", CreateListener.class).iterator();
        while (i.hasNext()) {
            widgetDefinition.addCreateListener((CreateListener)i.next());
        }
    }

    public void setWidgetDefinitionBuilders( Map widgetDefinitionBuilders )
    {
        this.widgetDefinitionBuilders = widgetDefinitionBuilders;
    }

    public void setWidgetValidatorBuilders( Map widgetValidatorBuilders )
    {
        this.widgetValidatorBuilders = widgetValidatorBuilders;
    }

    public void setWidgetListenerBuilders( Map widgetListenerBuilders )
    {
        this.widgetListenerBuilders = widgetListenerBuilders;
    }

    public void setDatatypeManager( DatatypeManager datatypeManager )
    {
        this.datatypeManager = datatypeManager;
    }

    public void setExpressionManager( ExpressionManager expressionManager )
    {
        this.expressionManager = expressionManager;
    }
}
